; this is a version of HUMAT which works for observable behaviours; for unobservable behaviours the knowledge about alters
; in the social networks will have to be represented differently, and this has an impact on dissonance reduction strategies
; (how do you calculate social satisfaction?, how do you choose who to get information from? and how do you choose who to try to convince?)
breed [humats humat]
breed [influencers influencer]
undirected-link-breed [connections connection]  ; connections among citizens
directed-link-breed [relations relation]        ; relations between citizens and authorites

globals [
  behavioural-alternatives
  A-supporters
  B-supporters
  A-value+       ; opponents with satisfaction
  A-value-       ; opponents with dissatisfaction
  A-value
  B-value+
  B-value-
  B-value
  group-number
  num-removed    ; number of groups that have been removed
  setup?         ; being setted up or not
  %enhancing-experiential-satisfation
  %enhancing-value-satisfaction
  num-authority
  norm-influence
]

influencers-own [; infuencers or authorities
  influence      ; for now random < 0 ; 1 > the higher, the less-or-more-vocalful
  endorsement    ; endorsing an alternative
]

connections-own [
  link-color
]
humats-own [
  behaviour
  ses            ; socio-economic status ; for now random < 0 ; 1 > the higher, the less-or-more-vocalful
  group-id
  tendency-for-change  ; relates to against-or-for-change
  vocie-being-heard    ; relates to vocie-being-heard

;;;dissonance-related variables;;; ;variables calculated for all BAs;
  experiential-importance
  social-importance
  values-importance
  non-social-importance

  experiential-satisfaction-A
  social-satisfaction-A
  values-satisfaction-A
  non-social-evaluation-A

  experiential-satisfaction-B
  social-satisfaction-B
  values-satisfaction-B
  non-social-evaluation-B

  experiential-evaluation-A ; evaluation of A (behavioural alternative i) with respect to experiential group of needs for HUMAT j <-1;1>
  social-evaluation-A       ; evaluation of A (behavioural alternative i) with respect to social group of needs for HUMAT j <-1;1>
  values-evaluation-A       ; evaluation of A (behavioural alternative i) with respect to values for HUMAT j <-1;1>
  experiential-evaluation-B ; evaluation of B (behavioural alternative ~i) with respect to experiential group of needs for HUMAT j <-1;1>
  social-evaluation-B       ; evaluation of B (behavioural alternative ~i) with respect to social group of needs for HUMAT j <-1;1>
  values-evaluation-B       ; evaluation of B (behavioural alternative ~i) with respect to values for HUMAT j <-1;1>

  evaluations-list-A
  evaluations-list-B
  evaluations-A
  evaluations-B
  satisfaction-A
  satisfaction-B
  satisfaction              ; the satisfaction of the decision
  support                   ; the degreen of the support
  satisfying-A              ; the sum of satisfying evaluations of A over three groups of needs <0;1,5>
  satisfying-B              ; the sum of satisfying evaluations of B over three groups of needs <0;1,5>
  dissatisfying-A           ; the sum of dissatisfying evaluations of A <0;1,5>
  dissatisfying-B           ; the sum of dissatisfying evaluations of B over three groups of needs <0;1,5>

  dissonance-A              ; the level of cognitive dissonance a behavioural alternative i (A) evokes in HUMAT j at time tn [Dij tn] <0;1>
  dissonance-B              ; the level of cognitive dissonance a behavioural alternative i (B) evokes in HUMAT j at time tn [Dij tn] <0;1>
  dissonance-tolerance      ; individual difference in tolerating dissonances before they evoke dissonance reduction strategies [Tj] normal trunc distribution with mean = 0.5, sd = 0.14 trunc to values <0;1>, this is the threshold determining if a reduction strategy is forgetting/distraction or if action has to be taken
  dissonance-strength-A     ; individually perceived strength of cognitive dissonance a behavioural alternative i (A) evokes in HUMAT j at time tn [Fij tn]; F because it's a fraction of maximum dissonance HUMAT j can possibly experience <0;1>
  dissonance-strength-B     ; individually perceived strength of cognitive dissonance a behavioural alternative i (B) evokes in HUMAT j at time tn [Fij tn]; F because it's a fraction of maximum dissonance HUMAT j can possibly experience <0;1>
  dissonance-strength       ; individually perceived strength of cognitive dissonance a chosen behavioural alternative evokes in HUMAT j at time tn [Fij tn]; either dissonance-strenght-A or dissonance-strength-B, depending on chosen behavioural alternative

  ;variables only calculated for the chosen BA, not for all BAs;
  experiential-dilemma?     ; the existence of an experiential dilemma e a chosen behavioural alternative i evokes in HUMAT j at time tn [Dije tn = {0,1}, where 0 donotes no dilemma and 1 denotes existence of dilemma]
  social-dilemma?           ; the existence of an social dilemma s a chosen behavioural alternative i evokes in HUMAT j at time tn [Dije tn = {0,1}, where 0 donotes no dilemma and 1 denotes existence of dilemma]
  values-dilemma?           ; the existence of an values dilemma v a chosen behavioural alternative i evokes in HUMAT j at time tn [Dije tn = {0,1}, where 0 donotes no dilemma and 1 denotes existence of dilemma]

;;;alter-representation variables;;;
  alter-representation-list
  inquiring-list
  signaling-list
  inquired-humat            ; the list belongs to ego and contains information about the alter who the ego inquires with
  inquiring-humat           ; the list belongs to an inquired alter and contains information about the ego who is inquiring
  signaled-humat            ; the list belongs to ego and contains information about the alter who the ego signals to
  signaling-humat           ; the list belongs to a signaled alter and contains information about the ego who is signaling

  inquiring?                ; boolean positive [1] if the ego is inquiring at a given tick
  #inquiring                ; the number of times humat inquired with alters
  #inquired                 ; the number of times humat was inquired with by egos
  signaling?                ; boolean positive [1] if the ego is signaling at a given tick
  #signaling                ; the number of times humat signaled to alters
  #signaled                 ; the number of times humat was signaled to

;;;;interaction record
  con-frequency             ; the frequence of interaction between the humat and (connected) others
]

;;;;;;;;;;;;;;;;;;
;;; Procedures ;;;
;;;;;;;;;;;;;;;;;;
to reset
  clear-all
  reset-ticks
  resize-world 0 200 0 200
  ask patches [set pcolor white]
  ;import-drawing "Picture 2.png"  ;background
  set group-number 1
  set num-removed 0
  set setup? false
end


;;;;;;;;;;;;;;;;;;;;;;;;
;;; Create the community ;;;
;;;;;;;;;;;;;;;;;;;;;;;;
to add-agents
 ifelse (group-number - num-removed) > num-groups
   [ user-message word (group-number - 1) " groups have been created!"]
   [ make-HUMATS
     to-show-labels?
   ]
  set norm-influence 0
end

to make-HUMATS
   random-seed 400                                        ; make the random process determined
   create-humats num-agents
    [
     set size 4 * (less-or-more-vocal + 2)
     set group-id group-number
     set label group-id
     set label-color black
     set dissonance-tolerance random-normal-trunc 0.5 0.14 0 1
     set tendency-for-change random-normal-trunc against-or-for-change 0.14 0 1
     set vocie-being-heard random-normal-trunc less-or-more-vocal 0.14 0 1
     set ses 1 * vocie-being-heard
    ]
  output-print (word "GROUP " group-number ":")
  output-print word "num-agents = " num-agents
  output-print word "less-or-more-vocal ≈ " less-or-more-vocal
  output-print word "against-or-for-change ≈ " against-or-for-change
  output-print "+++++++++++++++++++++++++"

  set group-number group-number + 1
  set-noun-social-need
  set-intra-networks                                      ; links within the same group
  set-social-satisfactions

  ask humats with [group-id = group-number - 1] [
   evaluate-satisfaction-support                          ; evaluate the support and satisfaction
   setxy (support * 37 + random 50)  (100 * satisfaction) ; agents' position relate to their support and satisfaction
  ]
end

to to-show-labels?
  ask humats [
    ifelse show-labels?
       [
        set label group-id
        set label-color black
        ]
       [set label ""]
   ]
end

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Setup Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;
to setup
  clear-all-plots
  reset-ticks
  set setup? true
  ask influencers [die]
  set %enhancing-experiential-satisfation 0
  set %enhancing-value-satisfaction 0
  set num-authority 0
  set-default-shape influencers "star"
  random-seed 300
  create-influencers num-authority [             ; create infuencers or authorities, or superspreaders
    setxy random-xcor (random-float (world-height - 1) / 4) + (world-height - 1) * 3 / 4
    set endorsement one-of (list "A" "B" )
    ifelse endorsement = "B"
      [set color green]
      [set color magenta]
    set size 10
    set influence [ycor] of self
  ]
  to-show-labels?
  set-inter-networks
  set-social-satisfactions
end

;;;;;;;;;;;;;;;;;;;;;;;
;;; Social networks ;;;
;;;;;;;;;;;;;;;;;;;;;;;
to set-intra-networks                            ; within the groups
  ask HUMATS with [group-id = group-number - 1] [
    let topological-neighbours other HUMATS with [group-id = group-number - 1]
    let #topological-neighbours count topological-neighbours

    create-connections-with n-of round (0.2 * #topological-neighbours) topological-neighbours
    if count connection-neighbors = 0
      [create-connection-with min-one-of other HUMATS [distance myself]]
     ]

  ask connections
    [set link-color 7
     set color link-color
     set thickness 0.01
     ]

  ifelse show-links?
    [ ask connections [set hidden? false]]
    [ ask connections [set hidden? true]]
end


to set-inter-networks                            ; links between the groups
 ask HUMATS [
    let self-group-id group-id
    let topological-neighbours other HUMATS with [group-id != self-group-id]
    let #topological-neighbours count topological-neighbours
    create-connections-with n-of round (0.05 * #topological-neighbours) topological-neighbours
             [ set link-color 7
               set color link-color
               set thickness  0.01
               set shape "inter-groups"
              ]
    set con-frequency n-values count humats [0]  ; set-up the list of connections of humat and other humats
    ]

  ask influencers [
    let topological-neighbours other HUMATS with [distance myself <= 150]
    let #topological-neighbours count topological-neighbours
    let link-color [color] of self + 3.5
    create-relations-to n-of (0.3 * #topological-neighbours) topological-neighbours
       [set color link-color
        set thickness  0.01]
    ]

  ifelse show-links?
    [ask relations [set hidden? false]
     ask connections [set hidden? false]
    ]
    [ask relations [set hidden? true]
     ask connections [set hidden? true]
    ]
end


to update-networks
  ask humats[
    let my-connections-with-other-humats my-connections with [[other-end] of myself != nobody]
    let other-humats [other-end] of my-connections-with-other-humats
    set other-humats turtle-set other-humats         ; should distinguish intra or inter-group links from here!!!
    let other-humats-ids sort [who] of other-humats
    let con-frequency-temp con-frequency

    let id1 [who] of self
    let groupID1 [group-id] of self

    let i 0
    while [ i < length other-humats-ids]
      [let id2 item i other-humats-ids
       let groupID2 [group-id] of (humat id2)
       ask connection id1 id2 [
         let greyscale item id2 con-frequency-temp
         let time-fading 0
         ifelse groupID1 = groupID2
            [set time-fading  time-voting * 1]
            [set time-fading  time-voting * 0.6]

         ifelse greyscale >= 0                                                    ; existing interactions
            [
             set link-color 7 / e ^ (0.2 * greyscale)
             ]
            [
             ifelse greyscale > (- time-fading)                                   ; (- time-for-vote * 0.8)
               [set link-color 3.9 / e ^ (0.2 * (greyscale + time-fading)) + 6]   ; link-color is 5, if no(greyscale = 0) interaction
               [set link-color 9.9
                set hidden? true
                ]
             ]
        set color link-color
        set thickness 0.05
        ]
       set i i + 1
       ]
     ]
end

to update-humats
 if (ticks > 0) and (ticks mod 5 = 0) and (norm-influence <= 0.8)
    [set norm-influence precision (norm-influence + 0.05) 2]

 ask humats [
    set evaluations-list-A (list (experiential-evaluation-A) (values-evaluation-A))
    set evaluations-list-B (list (experiential-evaluation-B) (values-evaluation-B))
    set non-social-evaluation-A sum evaluations-list-A / 2                      ; non-social satisfaction for the support
    set non-social-evaluation-B sum evaluations-list-B / 2
    set con-frequency (map [x -> x - 1 ] con-frequency)                         ; update the overall connections-frequency
    ifelse norm-influence > 0
      [set social-importance random-normal-trunc norm-influence 0.14 0 1]
      [set social-importance 0]
    evaluate-satisfaction-support
   ]
  move-humats
end

to move-humats
  ask humats [
     setxy (support * 37 + random 50)  (100 * satisfaction)
   ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Needs-and-satisfactions ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to set-noun-social-need
; in step 1, HUMATS set their initial behavior with the esclusion of social influence, i.e. on the basis their experiential and value satisfactions
  ask HUMATS with [group-id = group-number - 1][
  ; STEP 1
    ; set importances
    set experiential-importance random-normal-trunc 0.5 0.14 0 1
    set values-importance random-normal-trunc 0.5 0.14 0 1
    ifelse norm-influence > 0
      [set social-importance random-normal-trunc norm-influence 0.14 0 1]
      [set social-importance 0]

    ;either opposing or supporting
    let aver-vocal-change (less-or-more-vocal + 3 * against-or-for-change) / 4  ; against or for change can be more Decisive at the beginning
    let support-value normalized-min-max aver-vocal-change 0 1 -1 1

    ; A opposing the plan
    set experiential-satisfaction-A random-normal-trunc 0 0.5 -1 1
    set values-satisfaction-A random-normal-trunc 0 0.5 -1 1

    ; B supporting the plan
    set experiential-satisfaction-B random-normal-trunc (support-value + %enhancing-experiential-satisfation / 100) 0.50 -1 1
    set values-satisfaction-B random-normal-trunc (support-value + %enhancing-value-satisfaction / 100) 0.50 -1 1

    ; set evaluations = importances * satisfactions ; excluding social dimension
    ; A
    set experiential-evaluation-A experiential-importance * experiential-satisfaction-A
    set values-evaluation-A values-importance * values-satisfaction-A
    ; B
    set experiential-evaluation-B experiential-importance * experiential-satisfaction-B
    set values-evaluation-B values-importance * values-satisfaction-B

    set evaluations-list-A (list (experiential-evaluation-A) (values-evaluation-A))
    set evaluations-list-B (list (experiential-evaluation-B) (values-evaluation-B))

    ;;non-social satisfaction for the support
    set non-social-evaluation-A precision (sum evaluations-list-A / 2) 4  ;[-1,1]
    set non-social-evaluation-B precision (sum evaluations-list-B / 2) 4

    ;;people's behavior based on experiential need and value (initial opinion)
    ifelse non-social-evaluation-A < non-social-evaluation-B
      [ set behaviour "B" ]
      [ set behaviour "A"]
 ]
end

to set-social-satisfactions
; in step 2, HUMATS add the social satisfaction to the calculation of satisfactions from various behavioural alternatives
  ask HUMATS [
    ; STEP 2 based on cognition, not on interaction
    ; create alter representation lists for each alter  ; to do:if behaviours unobservable - here add the guessing parameter
    set-alter-representations
    alter-similarity
    calculate-dissonance
    calculate-dilemmas
 ]
end


to set-alter-representations
  ask HUMATS [
    set alter-representation-list []
    foreach sort connection-neighbors [x ->
      let working-list ( list
        [who] of x                              ;item 0 who
        0                                       ;item 1 inquired? 0 for not inquired with, 1 for inquired with already
        0                                       ;item 2 signaled? 0 for not signaled to, 1 for signaled to already
        [behaviour] of x                        ;item 3 behaviour ; to do: as known for observable and as guessed for non-observable behaviours
        same-BA? behaviour [behaviour] of x     ;item 4 1 for same behaviour; 0 for different behaviour ; used for inquiring
        ifelse-value (same-BA? behaviour [behaviour] of x  = 1) [0] [1] ; item 5  1 for different behaviour and 0 for same behaviour; used for signaling - coding is the other way around for the purpose of using the same reporter for sorting lists as in the case of inquiring
        0                                       ;item 6 [experiential-importance] of x
        0                                       ;item 7 [social-importance] of x
        0                                       ;item 8 [values-importance] of x
        0                                       ;item 9 [experiential-satisfaction-A] of x
        0                                       ;item 10 [social-satisfaction-A] of x
        0                                       ;item 11 [values-satisfaction-A] of x
        0                                       ;item 12 [experiential-satisfaction-B] of x
        0                                       ;item 13 [social-satisfaction-B] of x
        0                                       ;item 14 [values-satisfaction-B] of x
        0                                       ;item 15 similarity-A-experiential-importance - similarity between he importance of needs; only > 0 if the given BA satisfies that group of needs in a similar direction for the alter and for the ego
        0                                       ;item 16 simiarity-A-values-importance
        0                                       ;item 17 similarity-A-experiential-importance - similarity between he importance of needs; only > 0 if the given BA satisfies that group of needs in a similar direction for the alter and for the ego
        0                                       ;item 18 similarity-B-experiential-importance
        0                                       ;item 19 relative-aspiration-influenced-ego - relative social appeal/status (aspiration characteristic) in case of inquiring
        0                                       ;item 20 relative-aspiration-influenced-alter - relative social appeal/status (aspiration characteristic) in case of signaling
        0                                       ;item 21 inquiring-persuasion-A-experiential
        0                                       ;item 22 inquiring-persuasion-A-values
        0                                       ;item 23 inquiring-persuasion-B-experiential
        0                                       ;item 24 inquiring-persuasion-B-values
        0                                       ;item 25 signaling-persuasion-A-experiential
        0                                       ;item 26 signaling-persuasion-A-values
        0                                       ;item 27 signaling-persuasion-B-experiential
        0                                       ;item 28 signaling-persuasion-B-values
        0                                       ;item 29 inquiring-persusasion
        0                                       ;item 30 signaling-persuasion
      )
      set alter-representation-list lput working-list alter-representation-list
    ]

    set inquiring-list sort-list alter-representation-list 1 4 29
    set signaling-list sort-list alter-representation-list 2 5 30
    set inquired-humat []
    set inquiring-humat []
    set signaled-humat []
    set signaling-humat []
  ]
end

to update-dissonances
 ; each tick, all humats recalculate their social satisfactions and final satisfactions, and dissonances, not only the one that changed his mind
 ; update social satisfaction on the basis of alter-representation list;update social evaluation;update satisfaction from BAs
 ; update social dimension: social satisfaction from BAs, evaluations of BAs
   alter-similarity
   calculate-dissonance
   calculate-dilemmas
end

to alter-similarity  ;;relates to social need
    ; go through alter-representation-list and count the alters, who behave similarily
    let #similar 0
    foreach alter-representation-list [x -> if item 4 x = 1 [set  #similar  #similar + 1]]

    ; go through alter-representation-list and count the alters, who behave dissimilarily
    let #dissimilar 0
    foreach alter-representation-list [x -> if item 5 x = 1 [set  #dissimilar  #dissimilar + 1]]

    let #alters count connection-neighbors
    let %similar #similar / #alters
    let %dissimilar #dissimilar / #alters

    ; set social dimension: social satisfaction from BAs, evaluations of BAs
    ifelse behaviour = "A"
    [
      set social-satisfaction-A normalized-min-max %similar 0 1 -1 1
      set social-satisfaction-B normalized-min-max %dissimilar 0 1 -1 1
     ]
    [
      set social-satisfaction-B normalized-min-max %similar 0 1 -1 1
      set social-satisfaction-A normalized-min-max %dissimilar 0 1 -1 1
     ]

    set social-evaluation-A social-importance * social-satisfaction-A
    set social-evaluation-B social-importance * social-satisfaction-B
    set evaluations-list-A (list (experiential-evaluation-A) (social-evaluation-A) (values-evaluation-A))
    set evaluations-list-B (list (experiential-evaluation-B) (social-evaluation-B) (values-evaluation-B))

    ; set final satisfactions from BAs at setup stage
    set evaluations-A precision (sum evaluations-list-A / 3) 4
    set evaluations-B precision (sum evaluations-list-B / 3) 4
    set satisfaction-A sum (list experiential-satisfaction-A social-satisfaction-A values-satisfaction-A) / 3
    set satisfaction-B sum (list experiential-satisfaction-B social-satisfaction-B values-satisfaction-B) / 3
end

to calculate-dissonance
    set dissatisfying-A dissatisfying-status-BA evaluations-list-A
    set satisfying-A satisfying-status-BA evaluations-list-A
    set dissatisfying-B dissatisfying-status-BA evaluations-list-B
    set satisfying-B satisfying-status-BA evaluations-list-B
    set dissonance-A dissonance-status-A satisfying-A dissatisfying-A
    set dissonance-B dissonance-status-B satisfying-B dissatisfying-B

   ; calculating the need for dissonance reduction - a BA invokes the need to reduce dissonance if the level of dissonance for BA exceeds the dissonance-threshold
   set dissonance-strength-A (dissonance-A - dissonance-tolerance) / (1 - dissonance-tolerance)
    if dissonance-strength-A < 0
     [set dissonance-strength-A 0] ; if the dissonance level a behavioural alternative i (A) [Dij tn] does not exceed the individual tolerance threshold of HUMAT j [Tj], HUMAT j does not experience any dissonance: [Dij tn] < [Tj] -> [Fij tn] = 0
   set dissonance-strength-B  (dissonance-B - dissonance-tolerance) / (1 - dissonance-tolerance)
    if dissonance-strength-B < 0    ;the greater the dissonance, the more unsatisfied with the choice.
     [set dissonance-strength-B 0]  ;when dissonance approaches to 0 or smaller than the dissonance-tolerance, the more satisfied with the choice

   ; setting dissonance strength and colours
   ifelse behaviour = "A"  ;;example, opposing the plan
    [
     set dissonance-strength dissonance-strength-A
     ]
    [
     set dissonance-strength dissonance-strength-B
     ]
end

to calculate-dilemmas
;;calculating dilemmas does not inlcude 0 (< / >), therefore the same situation cannot evoke two different dilemmas
   ifelse behaviour = "A"
    [
     if (experiential-evaluation-A > 0 and social-evaluation-A < 0 and values-evaluation-A < 0)  or
        (experiential-evaluation-A <= 0 and social-evaluation-A >= 0 and values-evaluation-A >= 0)
        [set experiential-dilemma? 1]
     if (social-evaluation-A > 0 and experiential-evaluation-A < 0 and values-evaluation-A < 0)  or
        (social-evaluation-A <= 0 and experiential-evaluation-A >= 0 and values-evaluation-A >= 0)
        [set social-dilemma? 1]
     if (values-evaluation-A > 0 and experiential-evaluation-A < 0 and social-evaluation-A < 0)  or
        (values-evaluation-A <= 0 and experiential-evaluation-A >= 0 and social-evaluation-A >= 0)
        [set values-dilemma? 1]
    ]
    [
     if (experiential-evaluation-B > 0 and social-evaluation-B < 0 and values-evaluation-B < 0)  or
        (experiential-evaluation-B <= 0 and social-evaluation-B >= 0 and values-evaluation-B >= 0)
        [set experiential-dilemma? 1]
     if (social-evaluation-B > 0 and experiential-evaluation-B < 0 and values-evaluation-B < 0)  or
        (social-evaluation-B <= 0 and experiential-evaluation-B >= 0 and values-evaluation-B >= 0)
        [set social-dilemma? 1]
     if (values-evaluation-B > 0 and experiential-evaluation-B < 0 and social-evaluation-B < 0)  or
        (values-evaluation-B <= 0 and experiential-evaluation-B >= 0 and social-evaluation-B >= 0)
        [set values-dilemma? 1]
    ]
end


to evaluate-satisfaction-support
  ; initial opinon
 if num-authority = 0
   [ ifelse evaluations-A < evaluations-B   ;; people's behavior based on three need satisfaction
      [ set behaviour "B" ]
      [ set behaviour "A"]
    ]

  if ticks > 0
    [choose-action]

  let abs-satisfaction 0
  if behaviour = "A"
     [set color magenta
      set dissonance-strength dissonance-strength-A
      set support normalized-min-max non-social-evaluation-A -1 1 0 2
      set satisfaction normalized-min-max satisfaction-A -1 1 0 2
      set abs-satisfaction satisfaction-A
      ]

  if behaviour = "B"
     [set color green
      set dissonance-strength dissonance-strength-B
      set support normalized-min-max non-social-evaluation-B -1 1 2 4
      set satisfaction normalized-min-max satisfaction-B -1 1 0 2
      set abs-satisfaction satisfaction-B
      ]                                ;; eveluate if they support or resist the plan

   let smile word "smile " group-id
   let sad word "sad " group-id
   let neutral word "neutral " group-id
   set abs-satisfaction abs abs-satisfaction

  ifelse abs-satisfaction <= 0.02      ;; middle/moderate area, three types of face
     [set shape neutral]
     [ifelse satisfaction > 1          ;; nomalized satisfaction
       [set shape smile]
       [set shape sad ]
      ]
end


to choose-action ; HUMAT-oriented
 ; The BA comparison dimensions include:
 ;* overall satisfaction - if similarly satisfying (+/- 0.2 = 10% of the theoretical satisfaction range <-1;1>), then
 ;* dissonance level - if similarily dissonant (+/- 0.1 = 10% of the theoretical dissonance range <0;1>), then
 ;* satisfaction on experiential need - if similarly satisfying on experiantial need (+/- 0.2 = 10% of the theoretical experiantial satisfaction range <-1;1>), then
 ;* random choice.
  random-seed 200
  ask HUMATS[
    let num-A 0
    let num-B 0
    ask connection-neighbors [
      ifelse behaviour = "A"
      [set num-A num-A + 1]
      [set num-B num-B + 1]
     ]

   let norm-test (random-float 1)
   let stubb-test (random-float 1)
   ifelse  norm-influence * 2 > norm-test  ;; norm-influence is more dominant
    [ifelse num-A <=  num-B
      [set behaviour "B"]
      [set behaviour "A"]
     ]
    [ifelse tendency-for-change  <=  stubb-test
      [choose-habit
      ]
      [ifelse further-comparison-needed? evaluations-A evaluations-B 2
        [compare-dissonances]
        [ifelse evaluations-A < evaluations-B
          [set behaviour "B"]   ; make color a reporter and set color to reporter
          [set behaviour "A"]
         ]
       ]
     ]
   ]
end

to compare-dissonances   ; HUMAT-oriented
  (ifelse further-comparison-needed? dissonance-A dissonance-B 1
    [compare-experiential-needs]
    [ifelse dissonance-A > dissonance-B
     [set behaviour "A"] ; make color a reporter and set color to reporter
     [set behaviour "B"]
    ]
  )
end

to compare-experiential-needs ; HUMAT-oriented
  (ifelse further-comparison-needed?  experiential-satisfaction-A  experiential-satisfaction-B 0.001
    [choose-randomly]
    [ifelse  experiential-satisfaction-A > experiential-satisfaction-B
     [set behaviour "A"]      ; make color a reporter and set color to reporter
     [set behaviour "B"]]
  )
end

to choose-randomly
   set behaviour one-of (list "A" "B")
end

to choose-habit               ; in such situation humat stays with the previous behaviour (habit formation)
   set behaviour behaviour    ; A represents opposing the plan
end

;;;;;;;;;;;;;;;;;;;;;;;
;;; Go Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;
to go
ifelse not setup?
  [ user-message " SETUP-COMMUNITY at First!"]
  [ every 0.2                 ; set the speed of running
    [
      if ticks >= time-voting [ stop ] ; time for ending the voting process
      inquire
      signal
      if num-authority > 0
        [follow-authority]
      update-humats
      eveluation
      tick
    ]
   ]
end


to inquire
  ;here conceptualising dissonance reduction strategies depending on dilemmas a HUMAT is experiencing
  ;during inquiring information flows uni-directionally from the alter, who was giving advice to the ego, who was asking for advice
  ;activating dissonance reduction strategies - this is pre-action dissonance reduction; post action dissonnace reduction is when the action is taken
  ;- in Groningen case this is a longer process of making up your mind before the vote; its after the vote when post-action dissonance reduction will have to take place if the vote did not go your way
; INQUIRING
  ;inquire strategy - seek information in social network to reduce cognitive dissonance via altering EGO’s knowledge structures
  ;update ego's representation of the inquired alter
  ask humats [
   set inquiring? 0
   if dissonance-strength > 0 and (values-dilemma? = 1 or experiential-dilemma? = 1)  [
    set inquiring? 1
    set #inquiring #inquiring + inquiring?
    set inquired-humat item 0 inquiring-list
    set inquired-humat replace-item 1 inquired-humat 1
    set inquired-humat replace-item 6 inquired-humat [experiential-importance] of humat item 0 inquired-humat
    set inquired-humat replace-item 7 inquired-humat [social-importance] of humat item 0 inquired-humat
    set inquired-humat replace-item 8 inquired-humat [values-importance] of humat item 0 inquired-humat
    set inquired-humat replace-item 9 inquired-humat [experiential-satisfaction-A] of humat item 0 inquired-humat
    set inquired-humat replace-item 10 inquired-humat [social-satisfaction-A] of humat item 0 inquired-humat
    set inquired-humat replace-item 11 inquired-humat [values-satisfaction-A] of humat item 0 inquired-humat
    set inquired-humat replace-item 12 inquired-humat [experiential-satisfaction-B] of humat item 0 inquired-humat
    set inquired-humat replace-item 13 inquired-humat [social-satisfaction-B] of humat item 0 inquired-humat
    set inquired-humat replace-item 14 inquired-humat [values-satisfaction-B] of humat item 0 inquired-humat

    ; need-similarity is symmetric - if ego influences alter to the degree of 40% (max), alter also influences ego to the degree of 40%
    set inquired-humat replace-item 15 inquired-humat need-similarity experiential-evaluation-A [experiential-evaluation-A] of humat item 0 inquired-humat experiential-importance [experiential-importance] of humat item 0 inquired-humat ;similarity-A-experiential-importance - similarity between he importance of needs; only > 0 if the given BA satisfies that group of needs in a similar direction for the alter and for the ego[experiential-satisfaction-A] of humat item 0 inquired-humat
    set inquired-humat replace-item 16 inquired-humat need-similarity values-evaluation-A [values-evaluation-A] of humat item 0 inquired-humat values-importance [values-importance] of humat item 0 inquired-humat
    set inquired-humat replace-item 17 inquired-humat need-similarity experiential-evaluation-B [experiential-evaluation-B] of humat item 0 inquired-humat experiential-importance [experiential-importance] of humat item 0 inquired-humat ;similarity-A-experiential-importance - similarity between he importance of needs; only > 0 if the given BA satisfies that group of needs in a similar direction for the alter and for the ego
    set inquired-humat replace-item 18 inquired-humat need-similarity values-evaluation-B [values-evaluation-B] of humat item 0 inquired-humat values-importance [values-importance] of humat item 0 inquired-humat                         ;similarity-B-experiential-importance

    ; relative-aspiration is not symmetric
    set inquired-humat replace-item 19 inquired-humat relative-aspiration ses [ses] of humat item 0 inquired-humat         ; relative social appeal/status (aspiration characteristic) in case of inquiring
    ; persuasion is a multiplication of need-similatiry and relative-aspiration, and is not symmetric (different values for inquiring and signaling)
    set inquired-humat replace-item 21 inquired-humat (item 15 inquired-humat * item 19 inquired-humat) ; inquiring-persuasion-experiential-A = relative social appeal/status * similarity-A-experiential (similarity of needs activated by a BA)
    set inquired-humat replace-item 22 inquired-humat (item 16 inquired-humat * item 19 inquired-humat) ; inquiring-persuasion-values-A
    set inquired-humat replace-item 23 inquired-humat (item 17 inquired-humat * item 19 inquired-humat) ; inquiring-persuasion-experiential-B
    set inquired-humat replace-item 24 inquired-humat (item 18 inquired-humat * item 19 inquired-humat) ; inquiring-persuasion-values-B

    set inquired-humat replace-item 29 inquired-humat (item 21 inquired-humat + item 22 inquired-humat + item 23 inquired-humat + item 24 inquired-humat) ; inquiring persuasion = sum of inquiring persuasions
    ; here not seting the signaling persuasion, not to disturb the sorting process artificially by choosing the alter already inquired with for signaling, without going through the entire list of alters
    ; to do: describe the effect this has on the dynamics?  this is to do with sorting and the 3rd prio - it has to stay 0 for the ego to go through the entire list of alters before going for the most persuasive one over and over again

    ;update satisfactions
    set experiential-satisfaction-A new-need-satisfaction-inquiring experiential-satisfaction-A 21 9  ; none persuasive * ego's satisfaction + persuasive * alter's satisfaction
    set values-satisfaction-A new-need-satisfaction-inquiring values-satisfaction-A 22 11
    set experiential-satisfaction-B new-need-satisfaction-inquiring experiential-satisfaction-B 23 12
    set values-satisfaction-B new-need-satisfaction-inquiring values-satisfaction-B 24 14

    ;to do: for non-observable behaviours, set item 3 , item 4 and item 5 to the actual behaviour, not to a guessed one
    update-experiential-and-values-evaluations
    update-dissonances

    ;update the interaction frequency among humats, both sides of interaction
    set con-frequency replace-item (item 0 inquired-humat) con-frequency ((item (item 0 inquired-humat) con-frequency) + 2)  ;both sides of interation + 2
    ; update alter's representation of the inquiring ego
    ; please note that alter's representation of the inquiring ego is only updated to the extent of the ego's new decision about the BA (items 3,4,5)
    ask humat item 0 inquired-humat [
     foreach alter-representation-list
        [ x ->
          if item 0 x = [who] of myself
          [set inquiring-humat item position x alter-representation-list alter-representation-list]
         ]
     set inquiring-humat replace-item 3 inquiring-humat [behaviour] of myself
     set inquiring-humat replace-item 4 inquiring-humat same-BA? behaviour [behaviour] of myself
     set inquiring-humat replace-item 5 inquiring-humat ifelse-value (same-BA? behaviour [behaviour] of myself  = 1) [0] [1]

     foreach alter-representation-list
        [ x ->
          if item 0 x = [who] of myself
          [set alter-representation-list replace-item position x alter-representation-list alter-representation-list inquiring-humat]
         ]
     set inquiring-humat[]
     set #inquired #inquired + 1

     ;update the interaction frequency among humats, update the interaction of both sides
     set con-frequency replace-item ([who] of myself) con-frequency ((item ([who] of myself) con-frequency) + 2)
     update-dissonances
     update-networks
    ]

    ;update inquiring-list
    set inquiring-list replace-item 0 inquiring-list inquired-humat

    ;update alter-representation-list by replacing the old representation (of alter the ego inquired with) with a new representation that has accurate knowledge about alters action, motives and status of alteady having been inquired with
    foreach alter-representation-list
      [x ->
        if item 0 x = item 0 item 0 inquiring-list
        [set alter-representation-list replace-item position x alter-representation-list alter-representation-list item 0 inquiring-list]
       ]

    ; inquiring-list sorted:
     ;(1) ascendingly by inquired? (not inquired with first), and
     ;(2) descendingly by same-BA? (same behaviour first).
    set inquiring-list sort-list alter-representation-list 1 4 29
    set inquired-humat []
     ;ask alter [set inquiring-humat []]
    ]
  ]
end

to signal
; SIGNALING
  ;signaling strategy - seek information in social network to reduce cognitive dissonance via altering ALTER’s knowledge structures
  ;during signaling information flows uni-directionally from the ego, who was giving advice to the alter, who was made to listed
  ;signaled-humat - the list belongs to ego and contains information about the alter who the ego signals to
  ;signaling-humat - the list belongs to a signaled alter and contains information about the ego who is signaling

 ask humats [
   set signaling? 0
   if dissonance-strength > 0 and social-dilemma? = 1  [
     set signaled-humat item 0 signaling-list ;ego's representation of an alter as a temporary list

     ;update alter's representation of ego
     ask humat item 0 signaled-humat [
        set #signaled #signaled + 1
        foreach alter-representation-list
        [ x ->
          if item 0 x = [who] of myself
          [set signaling-humat item position x alter-representation-list alter-representation-list]
         ]
        set signaling-humat replace-item 3  signaling-humat [behaviour] of myself
        set signaling-humat replace-item 4  signaling-humat same-BA? behaviour [behaviour] of myself
        set signaling-humat replace-item 5  signaling-humat ifelse-value (same-BA? behaviour [behaviour] of myself  = 1) [0] [1]
        set signaling-humat replace-item 6  signaling-humat [experiential-importance] of myself
        set signaling-humat replace-item 7  signaling-humat [social-importance] of myself
        set signaling-humat replace-item 8  signaling-humat [values-importance] of myself
        set signaling-humat replace-item 9  signaling-humat [experiential-satisfaction-A] of myself
        set signaling-humat replace-item 10 signaling-humat [social-satisfaction-A] of myself
        set signaling-humat replace-item 11 signaling-humat [values-satisfaction-A] of myself
        set signaling-humat replace-item 12 signaling-humat [experiential-satisfaction-B] of myself
        set signaling-humat replace-item 13 signaling-humat [social-satisfaction-B] of myself
        set signaling-humat replace-item 14 signaling-humat [values-satisfaction-B] of myself

        ; need similarity (same values for inquiring and signaling)
        set signaling-humat replace-item 15 signaling-humat need-similarity experiential-evaluation-A [experiential-evaluation-A] of myself experiential-importance [experiential-importance] of myself ;similarity-A-experiential-importance - similarity between he importance of needs; only > 0 if the given BA satisfies that group of needs in a similar direction for the alter and for the ego[experiential-satisfaction-A] of humat item 0 signaled-humat
        set signaling-humat replace-item 16 signaling-humat need-similarity values-evaluation-A [values-evaluation-A] of myself values-importance [values-importance] of myself
        set signaling-humat replace-item 17 signaling-humat need-similarity experiential-evaluation-B [experiential-evaluation-B] of myself experiential-importance [experiential-importance] of myself ;similarity-A-experiential-importance - similarity between he importance of needs; only > 0 if the given BA satisfies that group of needs in a similar direction for the alter and for the ego
        set signaling-humat replace-item 18 signaling-humat need-similarity values-evaluation-B [values-evaluation-B] of myself values-importance [values-importance] of myself                             ;similarity-B-experiential-importance

        ; relative aspiration is not symmetric (different values for inquiring and signaling)
        set signaling-humat replace-item 20 signaling-humat relative-aspiration [ses] of myself ses         ; relative social appeal/status (aspiration characteristic) in case of signaling

        ; persuasion is a multiplication of need-similatiry and relative-aspiration, and is not symmetric (needs different values for inquiring and signaling)
        set signaling-humat replace-item 25 signaling-humat (item 15 signaling-humat * item 20 signaling-humat) ; signaling-persuasion-experiential-A = relative social appeal/status * similarity-A-experiential (similarity of needs activated by a BA)
        set signaling-humat replace-item 26 signaling-humat (item 16 signaling-humat * item 20 signaling-humat) ; signaling-persuasion-values-A
        set signaling-humat replace-item 27 signaling-humat (item 17 signaling-humat * item 20 signaling-humat) ; signaling-persuasion-experiential-B
        set signaling-humat replace-item 28 signaling-humat (item 18 signaling-humat * item 20 signaling-humat) ; signaling-persuasion-values-B

        ; seting new experiential and values satisfaction
        set experiential-satisfaction-A new-need-satisfaction-signaling experiential-satisfaction-A 25
        set values-satisfaction-A new-need-satisfaction-signaling values-satisfaction-A 26
        set experiential-satisfaction-B new-need-satisfaction-signaling experiential-satisfaction-B 27
        set values-satisfaction-B new-need-satisfaction-signaling values-satisfaction-B 28

        ; update alter's representation of the inquiring ego
        ; long-term memory - please note that alter's representation of the signaling ego only influences the update on social satisfaction, storing information other than about the BA (items 3, 4 aand 5) does not influence anything
        foreach alter-representation-list
          [x ->
            if item 0 x = [who] of myself
            [set alter-representation-list replace-item position x alter-representation-list alter-representation-list signaling-humat]
           ]

    ;update the interaction frequency among humats
    set con-frequency replace-item ([who] of myself) con-frequency ((item ([who] of myself) con-frequency) + 2)

    ;to do: sort the names and updating dissonances
        update-experiential-and-values-evaluations
        update-dissonances
      ]

    ;update the interaction frequency among humats
    set con-frequency replace-item (item 0 signaled-humat) con-frequency ((item (item 0 signaled-humat) con-frequency) + 2)

    ; update ego's representation of the signaled alter
    ; please note that alter's representation of the signaled alter is only updated to the extent of the alter's new decision about the BA (items 3,4,5)
    set signaled-humat replace-item  2 signaled-humat 1 ;representing the fact that alter was signaled to
    set signaled-humat replace-item  3 signaled-humat [behaviour] of humat item 0 signaled-humat
    set signaled-humat replace-item  4 signaled-humat same-BA? behaviour [behaviour] of humat item 0 signaled-humat
    set signaled-humat replace-item  5 signaled-humat ifelse-value (same-BA? behaviour [behaviour] of humat item 0 signaled-humat = 1) [0] [1]
    set signaled-humat replace-item 30 signaled-humat (item 25 [signaling-humat] of humat item 0 signaled-humat + item 26 [signaling-humat] of humat item 0 signaled-humat + item 27 [signaling-humat] of humat item 0 signaled-humat + item 28 [signaling-humat] of humat item 0 signaled-humat) ; gullibility (the willingness to be persuaded by the ego) of the alter is stores as the ego representation and is used later for sorting the signaling-list

    ask humat item 0 signaled-humat [
         set signaling-humat []
      ]

    ; update signaling-list
    set signaling-list replace-item 0 signaling-list signaled-humat

    ;update alter-representation-list by replacing the old representation (of alter the ego signaled to) with a new representation that has accurate knowledge about alters action and status of alteady having been signaled to
    foreach alter-representation-list
      [x ->
        if item 0 x = item 0 item 0 signaling-list
        [set alter-representation-list replace-item position x alter-representation-list alter-representation-list item 0 signaling-list]
       ]

    ; signaling-list sorted:
    set signaling-list sort-list alter-representation-list 2 5 30
    set signaled-humat []
    set signaling? 1
    set #signaling #signaling + 1
    update-dissonances
    update-networks
    ]
  ]
end

to follow-authority
  ;;the possibility of humats following the authority's endorsement
  ask humats [
     let num-influencer count relation-neighbors
     let influence-test (random 10)
     let idol 0
     ifelse num-influencer > 1
      [ let max-influence 0
        let influence-list []
        ask relation-neighbors [
           set influence-list lput influence influence-list
        ]
        set max-influence max influence-list
        set idol influencers with [influence = max-influence]
        if max-influence * 10 > influence-test
          [set behaviour [endorsement] of idol]
       ]
      [ let influence-temp 0
        ask relation-neighbors [ set influence-temp influence]
        set idol relation-neighbors
        if influence-temp * 10 > influence-test
         [set behaviour [endorsement] of idol
         ]
       ]
    set behaviour item 0 behaviour
   ]
end

to update-experiential-and-values-evaluations
  ; update evaluations = importances * satisfactions ; excluding social dimension
  ; A
  set experiential-evaluation-A experiential-importance * experiential-satisfaction-A
  set values-evaluation-A values-importance * values-satisfaction-A
  ; B
  set experiential-evaluation-B experiential-importance * experiential-satisfaction-B
  set values-evaluation-B values-importance * values-satisfaction-B

end

to-report random-normal-trunc [mid dev mmin mmax]
  ; creating a trunc normal function to be used for tuncating the normal distribution between mmin and mmax values
  ; good for now, but the goeal would be to get to the normal from beta (using gamma as a start)
  let result random-normal mid dev
  if result < mmin or result > mmax
  [report random-normal-trunc mid dev mmin mmax]
  report result
end

to-report random-trunc [randfloat mmin mmax]
  let result random-float randfloat * (1 / (0.14 * sqrt (2 * pi)))
  if result < mmin or result > mmax
  [report 1]
  report result
end

to-report normalized-min-max [norm-variable min-old max-old min-new max-new]
  let norm min-new + (((norm-variable - min-old) * (max-new - min-new)) / (max-old - min-old))
  report precision norm 4
end

; dissatisfying-status-BA and satisfying-status-BA should be merged into one reposrting procedure, however it seems that operators cannot be inputs in reporting procedures
to-report dissatisfying-status-BA [evaluations-list-BA]
 let dissatisfying-list-BA filter [i -> i < 0] evaluations-list-BA
 let dissatisfying-stat-BA abs sum dissatisfying-list-BA
 report dissatisfying-stat-BA
end

to-report satisfying-status-BA [evaluations-list-BA]
 let satisfying-list-BA filter [i -> i > 0] evaluations-list-BA
 let satisfying-stat-BA abs sum satisfying-list-BA
 report satisfying-stat-BA
end

to-report dissonance-status-A [sat-A dis-A] ; sat-A is satisfying-A and dis-A is dissatisfying-A
  let dissonant-A min (list sat-A dis-A)
  let consonant-A max (list sat-A dis-A)    ; in case of the same values, it does not matter
  let dissonance-of-A (2 * dissonant-A)/(dissonant-A + consonant-A)
  report dissonance-of-A
end

to-report dissonance-status-B [sat-B dis-B] ; sat-B is satisfying-B and dis-B is dissatisfying-B
  let dissonant-B min (list sat-B dis-B)
  let consonant-B max (list sat-B dis-B)    ; in case of the same values, it does not matter
  let dissonance-of-B (2 * dissonant-B)/(dissonant-B + consonant-B)
  report dissonance-of-B
end

to-report shading [dissonance-str]
 let shade normalized-min-max (-1 * dissonance-str) -1 0 -5 0
 report shade
end

to-report further-comparison-needed? [comparison-dimension-A comparison-dimension-B theoretical-range]
  let value 0
  ifelse (comparison-dimension-A > comparison-dimension-B - 0.1 * theoretical-range )
      and (comparison-dimension-A < comparison-dimension-B + 0.1 * theoretical-range )
  [set value true]
  [set value false]
  report value
end

to-report relative-aspiration [aspiration-influencing aspiration-influenced] ; produces values <0 ; 1 > - weighing of the influenced agent's status
  ; for inquiring the influencing agent is the alter, who is influencing the ego
  ; for signlaing the influencing agent is the ego, who is influencing the alter
  let rel-aspiration 0.4 + aspiration-influencing - aspiration-influenced
  if 0.4 + aspiration-influencing - aspiration-influenced > 1 [set rel-aspiration 1]
  if 0.4 + aspiration-influencing - aspiration-influenced < 0 [set rel-aspiration 0]
  report precision rel-aspiration 3
end

; to do: relative-ses-signaling [ego-ses alter-ses] ;still???
to-report same-BA? [ego-val alter-val]
  ifelse ego-val = alter-val
  [report 1] ; 1 is the same BA
  [report 0] ; 0 is different BA
end

;to do: add presuasion to sorting
to-report sort-list [the-list prio-1 prio-2 prio-3] ; the same sorting reporter for inquiring and signaling lists, just used with different priority characteristics
  ; the-list = the nested list you want to sort ; prio-1 = the first prio characteristic; prio-2 = the second prio characteristic; prio-3 - the third prio characteristic
  let sorted sort-by [[?1 ?2] -> item prio-3 ?1 > item prio-3 ?2] the-list
  set sorted sort-by [[?1 ?2] -> item prio-2 ?1 > item prio-2 ?2] sorted
  set sorted sort-by [[?1 ?2] -> item prio-1 ?1 < item prio-1 ?2] sorted
  report sorted
end

to-report need-similarity [need-evaluation-BA-ego need-evaluation-BA-alter need-importance-ego need-importance-alter] ; weighing of alter's similarity of needs, applicable to each group of needs for each BA
  ; can take a max value of 0,4 - if two agents value the same needs to the same extent, the influencing agent affects the influenced agent to a max degree of 40% (new value is 60% inflenced agent's and 40% influencing agent).
  ; if two agents don't find the same needs important, the influencing agent does not affect the influenced agent
  ifelse
  (need-evaluation-BA-ego > 0 and need-evaluation-BA-alter > 0) or
  (need-evaluation-BA-ego < 0 and need-evaluation-BA-alter < 0)
  [report 0.4 * (1 - abs(need-importance-ego - need-importance-alter))]
  [report 0]
end

to-report new-need-satisfaction-inquiring [need-satisfaction-BA #item #item2]
  ; the #item refers to the number of item on the list, which designates inquiring persuasion for each need-satisfaction-BA
  ; the #item2 refers to the number of item on the list, which designates need-satisfaction-BA of alter
  ; done for experiential needs and values of both BAs
  let val (1 - item #item inquired-humat) * need-satisfaction-BA + item #item inquired-humat * item #item2 inquired-humat
  report val
end

; to do: whis is to take into account that signaled-humat is the convincing's humat list and here it assumes to be the convinced's humat list - to change
to-report new-need-satisfaction-signaling [need-satisfaction-BA #item]
  ; the #item refers to the number of item on the list, which designates signaling persuasion for each need-satisfaction-BA
  ; when humats are persuaded by other humats in their social networks, they change their satisfactions of needs for BAs to the extent that the alter is persuasive (status * similar of needs importances for the BA)
  ; reports a new value of needs satisfaction for a persuaded HUMAT
  ; done for experiential needs and values of both BAs
  report (1 - item #item signaling-humat) * need-satisfaction-BA + item #item signaling-humat * [need-satisfaction-BA] of myself
end


to-report ns [need-satisfaction-BA]
  let ns-alter [need-satisfaction-BA] of humat item 0 inquired-humat
  report ns-alter
end

to-report consistency             ;clustering, to gauge the consistency between the humat and their neighbors
  let act-behaviour 0
  let nr-same 0
  let nr-neighbors 0
  ask humats [
    set act-behaviour behaviour
    ask connection-neighbors [
      set nr-neighbors nr-neighbors + 1
      if behaviour = act-behaviour
      [set nr-same nr-same + 1]
    ]
  ]
ifelse nr-neighbors = 0
  [report 0]
  [report precision (nr-same / nr-neighbors) 2]
end

to eveluation
  let list-experiential-A []
  let list-social-A []
  let list-values-A []
  let list-experiential-B []
  let list-social-B []
  let list-values-B []
  let A-supporters-temp 0
  let B-supporters-temp 0

  ask humats [
    set list-experiential-A lput experiential-satisfaction-A list-experiential-A
    set list-social-A lput social-satisfaction-A list-social-A
    set list-values-A lput values-satisfaction-A list-values-A
    set list-experiential-B lput experiential-satisfaction-B list-experiential-B
    set list-social-B lput social-satisfaction-B list-social-B
    set list-values-B lput values-satisfaction-B list-values-B

    set non-social-importance (experiential-importance + values-importance) / 2
    ifelse behaviour = "A"
     [set A-supporters-temp A-supporters-temp + 1]
     [set B-supporters-temp B-supporters-temp + 1]
  ]
  set A-supporters  A-supporters-temp
  set B-supporters  B-supporters-temp
  set A-value+ count humats with [behaviour = "A" and satisfaction-A > 0.02]
  set A-value- count humats with [behaviour = "A" and satisfaction-A < -0.02]
  set A-value count humats with [behaviour = "A" and satisfaction-A <= 0.02 and satisfaction-A >= -0.02]

  set B-value+ count humats with [behaviour = "B" and satisfaction-B > 0.02]
  set B-value- count humats with [behaviour = "B" and satisfaction-B < -0.02]
  set B-value count humats with [behaviour = "B" and satisfaction-B <= 0.02 and satisfaction-B >= -0.02]
end
@#$#@#$#@
GRAPHICS-WINDOW
347
17
958
629
-1
-1
3
1
12
1
1
1
0
0
0
1
0
200
0
200
1
1
1
ticks
60

BUTTON
19
510
269
543
SETUP-COMMUNITY
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
20
551
141
584
GO ONCE
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
972
266
1091
311
% OPPONENTS
precision ((count humats with [behaviour = \"A\"] / count humats )* 100) 0
17
1
11

BUTTON
147
551
269
584
GO
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
20
207
271
240
num-groups
num-groups
0
10
2
1
1
NIL
HORIZONTAL

MONITOR
1102
265
1222
310
% SUPPORTERS
precision ((count humats with [behaviour = \"B\"] / count humats )* 100) 0
17
1
11

PLOT
972
317
1223
468
SUPPORTER-or-OPPONENT?
NIL
NIL
0
10
0
10
true
true
"" ""
PENS
"Opponents" 1 0 -5825686 true "" "plot A-supporters"
"Supporters" 1 0 -10899396 true "" "plot B-supporters"

TEXTBOX
879
637
961
655
Support
13
0
1

TEXTBOX
285
601
342
621
Unhappy
13
0
1

TEXTBOX
298
24
339
42
Happy
13
0
1

TEXTBOX
641
637
727
656
Neutral
13
0
1

TEXTBOX
359
636
487
654
Oppose
13
0
1

TEXTBOX
19
184
259
202
STEP 2: Setting the community
14
14
1

TEXTBOX
19
292
238
311
STEP 3: Setting the group
14
14
1

OUTPUT
971
108
1222
231
14

TEXTBOX
1038
87
1177
105
Group Information
14
14
1

BUTTON
75
431
202
464
ADD-AGENTS
ADD-AGENTS
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
18
351
269
384
less-or-more-vocal
less-or-more-vocal
0
1
0.6
0.1
1
NIL
HORIZONTAL

SLIDER
18
313
269
346
against-or-for-change
against-or-for-change
0
1
0.8
0.1
1
NIL
HORIZONTAL

SLIDER
18
390
269
423
num-agents
num-agents
0
100
20
5
1
NIL
HORIZONTAL

BUTTON
82
135
203
168
RESET
reset
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
19
113
218
131
STEP 1: Reset the model
14
14
1

TEXTBOX
21
489
262
507
STEP 4: Running the model
14
14
1

PLOT
973
477
1223
628
HAPPY(+), NEUTRAL() or UNHAPPY(-)?
NIL
NIL
0
10
0
10
true
true
"" ""
PENS
"Opponents+" 1 0 -5825686 true "" "plot A-value+"
"Opponents-" 1 0 -2064490 true "" "plot A-value-"
"Opponents" 1 0 -534828 true "" "plot A-value"
"Supporters+" 1 0 -10899396 true "" "plot B-value+"
"Supporters-" 1 0 -13840069 true "" "plot B-value-"
"Supporters" 1 0 -2754856 true "" "plot B-value"

TEXTBOX
291
313
339
331
Neutral
13
0
1

SLIDER
20
245
271
278
time-voting
time-voting
50
200
80
10
1
NIL
HORIZONTAL

SWITCH
971
43
1094
76
show-links?
show-links?
0
1
-1000

SWITCH
1097
43
1220
76
show-labels?
show-labels?
1
1
-1000

TEXTBOX
19
20
264
100
NOTE: Initiate and formulate experimental scenarios using the INCITE-DEM dialogue tool by following these four steps.
16
14
1

TEXTBOX
973
20
1123
38
MODEL FEATURE:
14
14
1

TEXTBOX
974
244
1124
262
SIMULATION RESULT:
14
14
1
@#$#@#$#@
## WHAT IS IT?
This model aims to replicate the evolution of opinions and behaviours on a communal plan over time. It also aims to foster community dialogue on simulation outcomes, promoting inclusivity and engagement. Individuals (referred to as  **"agents"**), grouped based on Sinus Milieus (Groh-Samberg et al., 2023), face a binary choice: support or oppose the plan. Motivated by experiential, social, and value needs (Antosz et al., 2019), their decision is influenced by how well the plan aligns with these fundamental needs.

## HOW IT WORKS

During the model's operation, agents decide whether to support a plan, designated as the **"green hat"**, or oppose it, denoted as the **"red hat"**. Their choice between supporting or opposing is determined by a calculation that takes into account both their satisfaction with these options and the influence of social norms. Agents' satisfaction can be categorised into three dimensions based on their needs: experiential, social, and value-driven.

* Experiential need refers to your daily experience. Does the plan make your life more comfortable, and does it save money for example?

* Social need refers to your (dis)agreement with others. Do most of the people you know share your position, or are you the outlier in the group, making you an outcast?

* Value-driven need refers to important beliefs you have. For example, you may be very concerned about the environment, or you may be concerned about economic welfare.

Individual opinions remain dynamic through interactions with connected neighbours until they vote on the plan. During interactions, individuals may either convince others or may be convinced by others, and they may switch between being in favour or against the plan.


## HOW TO USE IT
To use this model, follow a four-step process. The initial three steps involve setting up the model: reset, setting the community, and setting the groups. Then, proceed to execute the model. Additional details are explained below.

### STEPS FOR IMPLEMENTATION AND ASSOCIATED COMMANDS AND PARAMETRES
To the left of the interface:

**Step 1: Reset the model**

The "RESET" completely clears all elements within the model's environment.

**Step 2: Setting the community**

During this step, establish the community based on specified characteristics. 

* **"num-groups"** determines the number of distinct groups within the simulated community, with a maximum value of 10 and a default setting of 2.

* **"time-voting"** specifies the duration for voting, setting the timeframe for model execution with a default value of 80.


**Step 3: Setting the group**

Groups have diverse properties, and agents within them vary. The model captures this diversity using two sliders for **"against-or-for-change"** and **"less-or-more-vocal"** characteristics to identify agents within groups.

* **"against-or-for-change"** reflects an agent's tendency to either maintain or change their behaviour, with values near 1 indicating openness to change and values near 0 signifying a preference for the status quo.

* **"less-or-more-vocal"** indicates an individual's vocal presence, where higher values suggest increased influence, with a value near 1 indicating strong vocal impact, while a value near 0 implies diminished vocal influence.

* **"num-agents"** allows the specification of the number of agents within a given group.

* **"ADD-AGENTS"** command enables the creation of agent groups by utilising selected parameters for "against-or-for-change," "less-or-more-vocal," and "num-agents." This combination allows the formation of diverse groups, such as a small outspoken faction in favor of a plan or a "silent majority" opposing a plan.

To form the extra group, adjust the three sliders as needed, then press the "ADD-AGENTS" button.

**Step 4: Running the model**

* **"SETUP-COMMUNITY"** command is employed to configure the entire system.

* **"GO-ONCE"** enables the model to execute for a single time step.

* **"GO"** function triggers ongoing execution of the model, enabling the generation of progress and outcomes based on the specified configurations.


### SYSTEM OUTPUT OVERVIEW
To the right of the interface:
#### MODEL FEATURE
* If the **"show-links?"** switch is on, the links between the agents become visible, showing the community structure; when the switch is off, links remain hidden.

* If **"show-labels?"** switch is on, the group ID becomes visible; when the switch is off, the group ID remains hidden.

* **"Group Information"** features an output box presenting detailed information specific to each group.

#### SIMULATION RESULTS

* The **"%OPPONENTS"** reveals the percentage of agents opposing a plan, while "%SUPPORTERS" indicates supporting agents, both evolving. 

* The **"SUPPORTERS-or-OPPONENT?"** graph depicts the temporal evolution of agent preferences between supporters (green hat) and opponents (red hat) within a specified timeframe.

* The **"HAPPY(+), NEUTRAL() or UNHAPPY(-)?"** chart gauges agent happiness based on their stance on the plan. Labels like "opponent+" signify happy opposing agents, "opponent-" indicates unhappy opposers, and "opponent" represents agents who oppose without a specific emotional state. Similarly, "supporters+", "supporters-", and "supporters" denote happiness, unhappiness, or neutrality among supporting agents.

## THINGS TO NOTICE

Emphasising the importance of maintaining static parameters during model execution, it is crucial to note that any changes to the remaining parameters should be avoided. 

Moreover, as the model operates, the connections among individuals become clearer, indicating a higher frequency of interactions. Conversely, weakened connections suggest a decrease in interaction frequency, while vanished connections signify prolonged absence of interaction.


## NETLOGO FEATURES

In this conceptual framework, the organisation of visual elements follows a designated symbolic structure:

* Individual agents are symbolised by **nodes or heads**.

* The relationships between these agents are represented by **connecting lines or edges**. "Dashed lines" signify connections between different groups, while "solid lines" depict connections within the same group.

* Distinct groups are identified by the **colour of each agent's face**.

* The specific opinions of agents are indicated by the **colour of their hats**, denoted as either "green" or "red."

* The importance of agents' vocal contributions is reflected in the **size of their representations**.

* **Facial expressions**, depicted as happy, sad, or neutral faces, convey the satisfaction, dissatisfaction, or indifference experienced by these agents.


## CREDITS AND REFERENCES

* Groh-Samberg O, Schröder T, Speer A. Social milieus and social integration. From theoretical considerations to an empirical model. KZfSS KZfSS Cologne Journal for Sociology and Social Psychology. 2023 Jul 5:1-25.

* Antosz P, Jager W, Polhill G, Salt D, Alonso-Betanzos A, Sánchez-Maroño N, Guijarro-Berdiñas B, Rodríguez A. Simulation model implementing different relevant layers of social innovation, human choice behaviour and habitual structures. SMARTEES Deliverable. 2019.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

neutral 1
false
0
Circle -2064490 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 10
false
0
Circle -1184463 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 2
false
0
Circle -13791810 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 3
false
0
Circle -955883 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 4
false
0
Circle -11221820 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 5
false
0
Circle -14835848 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 6
false
0
Circle -13345367 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 7
false
0
Circle -8630108 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 8
false
0
Circle -13840069 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

neutral 9
false
0
Circle -6459832 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Rectangle -16777216 true false 105 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sad 1
false
0
Circle -2064490 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 10
false
0
Circle -1184463 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 2
false
0
Circle -13791810 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 3
false
0
Circle -955883 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 4
false
0
Circle -11221820 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 5
false
0
Circle -14835848 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 6
false
0
Circle -13345367 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 7
false
0
Circle -8630108 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 8
false
0
Circle -13840069 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

sad 9
false
0
Circle -6459832 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 210 105 225 90 240 90 240 90 255 90 262 120 240 150 225 192 247 210 262 210 255 210 240 210 240 195 225
Polygon -7500403 true true 150 15 45 120 255 120

smile 1
false
0
Circle -2064490 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 10
false
0
Circle -1184463 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 2
false
0
Circle -13791810 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 3
false
0
Circle -955883 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 4
false
0
Circle -11221820 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 5
false
0
Circle -14835848 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 6
false
0
Circle -13345367 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 7
false
0
Circle -8630108 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 8
false
0
Circle -13840069 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

smile 9
false
0
Circle -6459832 true false 60 105 180
Circle -16777216 true false 105 150 30
Circle -16777216 true false 165 150 30
Polygon -16777216 true false 150 255 105 240 90 225 90 225 90 210 90 203 120 225 150 240 192 218 210 203 210 210 210 225 210 225 195 240
Polygon -7500403 true true 150 15 45 120 255 120

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0
-0.2 0 0 1
0 1 1 0
0.2 0 0 1
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

inter-groups
0
-0.2 0 0 1
0 1 4 4
0.2 0 0 1
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@

@#$#@#$#@
